# -*- coding:utf-8 -*-
import os
import sys
import re

def extract_prefix_and_parts(filename):
    """
    解析文件名，提取编号前缀、中文部分和扩展名。
    针对 yontenbzangpo0703_信仰吟诵.c3.7s.txt 这种格式
    返回 (prefix, chinese_part, extension) 或 None。
    """
    # 匹配格式：字母数字前缀_中文部分.扩展名
    pattern = r'^([a-zA-Z0-9_]+)_([^.]+)(\..*?)$'
    match = re.match(pattern, filename)
    if match:
        prefix = match.group(1)  # 如 yontenbzangpo0703
        chinese_part = match.group(2)  # 如 信仰吟诵 或 继续第七函的翻译
        extension = match.group(3)  # 如 .c3.7s.txt 或 .txt
        return prefix, chinese_part, extension
    return None

def find_all_files_recursively(directory):
    """
    递归遍历目录，返回所有文件的完整路径列表。
    """
    all_files = []
    try:
        for root, dirs, files in os.walk(directory):
            for file in files:
                all_files.append(os.path.join(root, file))
    except Exception as e:
        print(f"遍历目录时出错：{e}")
    return all_files

def unify_filenames_in_directory(target_dir):
    """
    遍历指定目录下的所有文件（包括子目录），将同一编号前缀下文件的中文部分
    统一为 .c3.7s 文件的中文部分。
    """
    if not os.path.isdir(target_dir):
        print(f"错误：目标目录 '{target_dir}' 不存在或不是一个目录。")
        return

    renamed_count = 0
    failed_count = 0

    print(f"\n开始递归处理目录：'{target_dir}'")

    # 获取所有文件的完整路径
    all_files = find_all_files_recursively(target_dir)
    print(f"找到 {len(all_files)} 个文件。")

    if len(all_files) == 0:
        print("目录中没有找到任何文件。")
        return

    # 收集所有文件的编号前缀和文件信息
    files_by_prefix = {}
    parsed_files = []
    
    for filepath in all_files:
        filename = os.path.basename(filepath)
        parts = extract_prefix_and_parts(filename)
        if parts:
            prefix, chinese_part, extension = parts
            parsed_files.append((filepath, filename, prefix, chinese_part, extension))
            if prefix not in files_by_prefix:
                files_by_prefix[prefix] = []
            files_by_prefix[prefix].append((filepath, filename, chinese_part, extension))
        else:
            print(f"  跳过无法解析的文件：'{filename}'")

    print(f"成功解析 {len(parsed_files)} 个文件。")
    print(f"找到 {len(files_by_prefix)} 个不同的编号前缀组。")

    if len(files_by_prefix) == 0:
        print("没有找到符合格式的文件。")
        return

    # 处理每个编号前缀组
    for prefix, file_list in files_by_prefix.items():
        print(f"\n处理编号前缀组：'{prefix}' ({len(file_list)} 个文件)")
        
        # 显示该组中的所有文件
        for filepath, filename, chinese_part, extension in file_list:
            print(f"  文件: {filename}")
        
        # 查找 .c3.7s 文件作为参考
        reference_chinese_part = None
        reference_filename = None
        for filepath, filename, chinese_part, extension in file_list:
            if '.c3.7s' in extension:
                reference_chinese_part = chinese_part
                reference_filename = filename
                print(f"  找到参考文件：'{filename}' -> 参考中文部分：'{chinese_part}'")
                break

        if not reference_chinese_part:
            print(f"  警告：编号前缀 '{prefix}' 没有找到 .c3.7s 文件作为参考。跳过此编号的文件。")
            failed_count += len(file_list)
            continue

        # 对该编号下的其他文件进行重命名
        for filepath, filename, chinese_part, extension in file_list:
            if '.c3.7s' in extension:
                print(f"  跳过参考文件：'{filename}'")
                continue  # 跳过参考文件本身

            # 生成新的文件名，使用参考文件的中文部分
            new_filename = f"{prefix}_{reference_chinese_part}{extension}"
            new_filepath = os.path.join(os.path.dirname(filepath), new_filename)

            # 如果新旧名称相同，则无需重命名
            if filepath == new_filepath:
                print(f"  文件名已一致，无需重命名：'{filename}'")
                continue

            # 检查目标路径是否已存在
            if os.path.exists(new_filepath):
                print(f"  警告：目标文件 '{new_filepath}' 已存在，跳过重命名 '{filename}'。")
                failed_count += 1
                continue

            try:
                os.rename(filepath, new_filepath)
                print(f"  ✓ 文件重命名成功：'{filename}' -> '{new_filename}'")
                renamed_count += 1
            except PermissionError:
                print(f"  ✗ 错误：无权限重命名文件 '{filepath}'。请检查权限。")
                failed_count += 1
            except OSError as e:
                print(f"  ✗ 错误：重命名文件 '{filepath}' 失败：{e}")
                failed_count += 1
            except Exception as e:
                print(f"  ✗ 错误：重命名文件 '{filepath}' 时发生未知错误：{e}")
                failed_count += 1

    print(f"\n重命名完成。成功重命名 {renamed_count} 个文件，{failed_count} 个文件失败。")

def main():
    # 警告提示
    print("\n" + "="*60)
    print("!!! 极度重要警告 !!!")
    print("此脚本将递归遍历指定目录中的所有文件并修改文件名，操作不可逆。")
    print("适用于格式：编号前缀_中文部分.扩展名")
    print("如：yontenbzangpo0703_信仰吟诵.c3.7s.txt")
    print("在继续之前，请务必：")
    print("1. **备份您的所有重要文件！**")
    print("2. **在一个包含测试文件的非重要目录中进行测试！**")
    print("3. **确保每个编号前缀组都有一个 .c3.7s 文件作为参考！**")
    print("="*60 + "\n")

    # 要求用户输入目标目录路径
    target_directory_input = input("请输入要处理的目录路径：")
    target_directory_input = target_directory_input.strip('\'"')

    # 确认目录是否存在
    if not os.path.isdir(target_directory_input):
        print(f"错误：您输入的目录 '{target_directory_input}' 不存在。请检查路径是否正确。")
        if sys.platform.startswith('win'):
            input("\n按任意键退出...")
        sys.exit(1)

    # 显示目录信息
    print(f"\n目标目录：'{target_directory_input}'")
    
    # 列出目录中的一些文件作为确认
    try:
        items = os.listdir(target_directory_input)
        print(f"目录中包含 {len(items)} 个项目")
        if len(items) > 0:
            print("前几个文件示例：")
            count = 0
            for item in items:
                item_path = os.path.join(target_directory_input, item)
                if os.path.isfile(item_path) and count < 5:
                    print(f"  {item}")
                    count += 1
    except Exception as e:
        print(f"无法读取目录内容：{e}")
        if sys.platform.startswith('win'):
            input("\n按任意键退出...")
        sys.exit(1)

    # 确认操作
    confirm = input("\n确定要继续吗？ (输入 'yes' 继续)：").lower()
    if confirm != 'yes':
        print("\n操作已取消。")
        if sys.platform.startswith('win'):
            input("\n按任意键退出...")
        sys.exit(0)

    # 执行重命名操作
    unify_filenames_in_directory(target_directory_input)

    print("\n所有重命名任务完成。")
    if sys.platform.startswith('win'):
        input("\n按任意键退出...")

if __name__ == "__main__":
    main()
